#include "util/device_sync_parser.h"

#include <argp.h>
#include <stdbool.h>
#include <string.h>
#include <sys/sysinfo.h>
#include <stdlib.h>

#include "util/generic_conf_parser.h"
#include "util/sync_control_parser.h"

#ifndef MANUAL_RELEASE_TAG
#include "manual_release_tag.h"
#endif

const char *device_sync_doc =
		"Device Synchronization - Waits until all devices and attributes defined in a given config file are available.";

#ifdef COMP_GIT_VERSION
#define VERSION_INFO_VERSION "2.0_" COMP_GIT_VERSION "(last update: " MANUAL_RELEASE_TAG ")"
#else
#define VERSION_INFO_VERSION "2.0_" MANUAL_RELEASE_TAG
#endif


static const char *device_sync_program_version = "device_sync, " VERSION_INFO_VERSION;

/*
 * A description of the arguments we accept.
 */
static const char * const device_sync_args_doc = "<conf file>";

/*
 * The options we understand.
 */
static const struct argp_option const device_sync_options[] = {
		{"timeout", 't', "<timeout>" , 0, "Exit after timeout [ms] elapsed even if the devices are still not there. (Default=1000)",0},
		{"debug", 'd', NULL, 0, "Prints debug messages on the console.",0},
		{"version", 'V', NULL, 0, "Print device sync version.",0},
		{NULL,0,NULL,0,NULL,0}
};

//-------------------------------------------- properties ------------------------------------------------------------
static unsigned int timeout_in_ms=DEFAULT_TIMEOUT;
//--------------------------------------------------------------------------------------------------------------------

//-------------------------------------------- private function declaration ------------------------------------------
static char *device_sync_parser_filter_help(int key, const char *text, void *input);

static error_t device_sync_parser_parse_opt (int key, char *arg, struct argp_state *state);
//--------------------------------------------------------------------------------------------------------------------

static const kml_conf_parser_t *parser_ary[]=
{
		&device_parser,
		&attribute_parser,
		NULL
};

//-------------------------------------------- public functions ------------------------------------------------------
logger_loglevel_t device_sync_parser_parse_loglevel(int argc, char *argv[])
{
	int i;
	for (i=1;i<argc;i++)
	{
		if (strcmp(argv[i],"-d")==0 || strcmp(argv[i],"--debug")==0)
			return LOGGER_LEVEL_DEBUG;
	}

	return LOGGER_LEVEL_ERROR;
}

error_code_t device_sync_parser_parse_args(int argc, char *argv[])
{
	error_code_t result=RESULT_OK;

	struct argp argp = {device_sync_options, device_sync_parser_parse_opt,
			device_sync_args_doc, device_sync_doc, NULL, device_sync_parser_filter_help,NULL};

	//initialize config parser
	generic_conf_parser_init(parser_ary);

	//parse command line options
	if (argp_parse(&argp, argc, argv, ARGP_NO_EXIT, 0, &result)==EINVAL)
		result=RESULT_INVALID_ARGS;

	logger_log_debug("DEVICE_SYNC_PARSER -> init done (result: %d).", result);
	return result;
}

error_code_t device_sync_parser_parse_config(void)
{
	if (generic_conf_parser_file_read_in())
		return generic_conf_parser_parse();
	else
		return RESULT_OK;
}

unsigned int device_sync_parser_get_timeout_ms(void)
{
	return timeout_in_ms;
}
//--------------------------------------------------------------------------------------------------------------------


//-------------------------------------------- private functions -----------------------------------------------------
static char *device_sync_parser_filter_help(int key, const char *text, void *input)
{
	error_code_t *result_ptr;
	result_ptr=(error_code_t *)input;

	if ((key & ARGP_KEY_HELP_POST_DOC)!=0)
		*result_ptr=RESULT_HELP_PRINTED;
	return (char *)text;
}

static error_t device_sync_parser_parse_opt (int key, char *arg, struct argp_state *state)
{
	int tmp_int;
	char* parse_int_result_ptr;
	error_code_t *result_ptr;
	result_ptr=(error_code_t *)state->input;

	switch (key)
	{
	case 'd':
		//we handled this attribute already at the beginning of main
		break;
	case 't':
		tmp_int=strtol(arg,&parse_int_result_ptr,10);
		if (parse_int_result_ptr==arg || tmp_int<=0)
		{
			logger_log_error("Invalid parameter. Option -t and --timeout must be followed by a positive number.");
			argp_usage(state);
			*result_ptr=RESULT_INVALID_ARGS;
		}
		else
			timeout_in_ms=(unsigned int)tmp_int;
		break;
	case 'V':
		printf("%s\n",device_sync_program_version);
		(*result_ptr)=RESULT_HELP_PRINTED;
		break;
	case ARGP_KEY_ARG:
		if (!generic_conf_parser_file_read_in())
		{
			*result_ptr=generic_conf_parser_read_file(arg);
		}
		else
		{
			logger_log_error("Only one configuration file is allowed to be passed as command line option.");
			argp_usage(state);
			*result_ptr=RESULT_INVALID_ARGS;
		}
		break;
	case ARGP_KEY_END:
		break;

	default:
		return ARGP_ERR_UNKNOWN;
	}
	return 0;
}
//--------------------------------------------------------------------------------------------------------------------
